/**
 *
 * \file        Hid.cpp
 *
 * \brief       Functions for transmitting HID data over IP
 *
 * \author      AKN
 *
 * \date        10/26/2008
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dm_string_utils.h>
#include <ctype.h>
#include "nutypedefs.h"
#include "Hid.h"
#include "console.h"
#include "hardware.h"
#include "os.h"
#include "ethernet.h"
#include "field_debug.h"
#include "errors.h"
#include "gateway_errors.h"
#include "StreamMgr.h"
#include "cresnet_slave.h"
#include "DeviceJoinInterface.h"

static void Hid_CloseTcpConnection(int code);
static void Hid_OpenTcpConnection(void);
/*static void * HID_Mouse_Repeat_Handler(UINT32 param);
static void * HID_Kbd_Repeat_Handler(UINT32 param);*/
UINT32 HIDConnectionTaskPtr;
UINT32 HidTcpTxSemaphore;
void HID_ConnectionTask(UINT32 param);
void HID_TcpConnectionCtrl(unsigned short usCmd);
INT32 HIDNewConnection(UINT32 param, UINT32 remoteIp, UINT16 remotePort, UINT32 socket);
INT32 HIDConn_Receive_Data(UINT32 param, UINT8 * pData, UINT32 dataBytes);
void HID_ConnectionInit(void);
int SendToHIDTCPServer( unsigned char* buffer, int buffLen);
//static unsigned char SetRelMouseButtonBits(INT8 u8ButtonState);
static UINT32 g_iHID_Sema = 0;

HIDCONN_PARAM InitialHidConnParam;
HIDCONN_PARAM *pHidConnParam;   // pointer to working copy for changes
HID_SWITCH_COMMAND hid_sw_cmd_st;    //Local HID switch info structure
HID_TCP_PARAM_St  g_stHIDServer, *g_pstHIDServer = NULL;
UINT8 g_MouseButtonState = NULL;
UINT8 g_KeyboardKeyState = NULL;
UINT8 g_KeyboardModState = NULL;
static bool g_bIgnoreTCP= false;

#ifdef DM_V24
bool ProjectEnterMKEvent(unsigned char* buffer, int buffLen);//Project specific functions
#endif

//Mouse Specific
#define MOUSE_LEFT_BUTTON     1
#define MOUSE_RIGHT_BUTTON    2
#define MOUSE_MIDDLE_BUTTON   4
#define REL_MOUSE_DEF_BYTE_0  0x81
#define MOUSE_REPEAT_TIME_MS  100

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      -
 * \brief       Initialze the HID conn params
 * \detail
 * \param       void
 */
void Init_HID(BOOL ignoreTCP)
{
    pHidConnParam = GetHidConnParams();


    OsCreateLock(&g_iHID_Sema);

    memcpy((char *)&InitialHidConnParam, (char *)pHidConnParam, sizeof(HIDCONN_PARAM));
    HID_ConnectionInit();
    g_bIgnoreTCP= ignoreTCP;
    return;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      INT32
 * \retval      0 if successful
 * \brief       Show or set the HID destination IP address
 * \detail      This is the destination where the mouse/Keyboard packets are sent
 * \param       UINT32 ignore
 * \param       char * cmd
 */
INT32 SetHIDAddress(UINT32 ignore, char * cmd)
{
    if( !cmd || !(*cmd) || (*cmd == '?') )                  // print help string
    {
        if ((cmd && *cmd == '?'))
        {
            DmConsolePrintf("HIDADDRESS [aaa.bbb.ccc.ddd]\r");
            DmConsolePrintf("where [aaa.bbb.ccc.ddd] is the IP address of the\r");
            DmConsolePrintf("destination server in dot notation.\r");
            DmConsolePrintf("HIDADDRESS with no arguments displays the current destination setting.\r");
        }
        else
        {
            unsigned char* cp = (unsigned char*) &pHidConnParam->HIDAddress;
            DmConsolePrintf("HID Destination Address: %u.%u.%u.%u\r",cp[0],cp[1],cp[2],cp[3]);
        }
    }
    else // parse the command line
    {
        unsigned long u[4];
        char *p;

        // convert dot decimal notation to 4 bytes
        p = strtok(cmd," .") ;
        for (int iByte = 0; iByte < 4; iByte++)
        {
            if ( !p )
                return -1;
            u[iByte]  = atoi(p);
            if (u[iByte] > 255)
                return iByte + 1;
            // get next token
            p = strtok(NULL," .") ;
        }
        // concatenate the bytes
        pHidConnParam->HIDAddress = (u[3] << 24) | (u[2] << 16) | (u[1] << 8) | u[0];
        /* Control the connection. Should be called after updating params */
        HID_TcpConnectionCtrl(HID_ADDR_CHANGED);
        // save the setting in NVL
        //TriggerSystemSave();
    }
    return 0;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      INT32
 * \retval      0 if successful
 * \brief       Show or set the HID connection type
 * \detail      connection type can be either TCP or UDP
 * \param       UINT32 ignore
 * \param       char * cmd
 */
INT32 SetHIDConnect(UINT32 ignore, char * cmd)
{
    char *TmpPtr;
    char ignoreStr[10];

    if (!g_bIgnoreTCP)
        sprintf(ignoreStr,"[TCP|");
    else
        sprintf(ignoreStr,"[");
    // if no arguments or argument is a ?,
    if( !cmd || !(*cmd) || (*cmd == '?') )                  // print help string
    {
        if ((cmd && *cmd == '?'))
        {
            DmConsolePrintf("HIDCONNecttype %sUDP|NONE]\r", ignoreStr);
            DmConsolePrintf("where %sUDP|NONE] is type of communication set\r", ignoreStr);
            DmConsolePrintf("and NONE if nothing is set.\r");
            DmConsolePrintf("HIDCONNECType with no arguments displays the current connection type.\r");
        }
        else
        {
            char name[5];

            //switch( InitialHidConnParam.HIDConnect)
            switch (pHidConnParam->HIDConnect)
            {
            case HID_CONNECTION_TCP:
                strcpy(&name[0], "TCP");
                break;
            case HID_CONNECTION_UDP:
                strcpy(&name[0], "UDP");
                break;
            case HID_CONNECTION_NONE:
            default:
                strcpy(&name[0], "NONE");
            }
            DmConsolePrintf("HID Connection type = %s\r",name);
        }
    }
    else // parse the command line
    {
        //Because this @#$%@^compiler does not support strcmpi!!!
        TmpPtr= cmd;
        while ( *TmpPtr != 0)
        {
            *TmpPtr = toupper(*TmpPtr);
            TmpPtr++;
        }

        if (!g_bIgnoreTCP && (!strcmp(cmd,"TCP")))
        {
            HID_TcpConnectionCtrl(HID_OPEN_TCP_CONN);
            pHidConnParam->HIDConnect = HID_CONNECTION_TCP; /* Should be after the function call */
        }
        else if (!strcmp(cmd,"UDP"))
        {
            HID_TcpConnectionCtrl(HID_OPEN_UDP_CONN);
            pHidConnParam->HIDConnect = HID_CONNECTION_UDP; /* Should be after the function call */
        }
        else if (!strcmp(cmd,"NONE"))
        {
            HID_TcpConnectionCtrl(HID_NO_CONN);
            pHidConnParam->HIDConnect = HID_CONNECTION_NONE;
        }
        else
        {
            //Bugzilla 25164
            DmConsolePrintf("HID Connection type is invalid; supported types are %sUDP|NONE]\r",ignoreStr);
            return -1;
        }
        //TriggerSystemSave();
        HID_TcpConnectionCtrl(HID_PORT_CHANGED);
    }
    return 0;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      INT32
 * \retval      0 if successful
 * \brief       Show or set the HID destination port
 * \detail
 * \param       UINT32 ignore
 * \param       char * cmd
 */
INT32 SetHIDPort(UINT32 ignore, char * cmd)
{
    // if no arguments or argument is a ?,
    if( !cmd || !(*cmd) || (*cmd == '?') )                  // print help string
    {
        if ((cmd && *cmd == '?'))
        {
            DmConsolePrintf("HIDPORT [portnum]\r");
            DmConsolePrintf("where 'portnum' is the port number\r");
            DmConsolePrintf("of destination server.\r");
            DmConsolePrintf("HIDPORT with no arguments displays the current destination port setting.\r");
        }
        else
        {
            DmConsolePrintf("HID Destination port = %u\r",pHidConnParam->HIDPort);
        }
    }
    else // parse the command line
    {
        long PortNumber;

        PortNumber = atoi(cmd);
        pHidConnParam->HIDPort = PortNumber;
        // save the setting in NVL
        //TriggerSystemSave();
        HID_TcpConnectionCtrl(HID_PORT_CHANGED);
    }
    return 0;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Control the connection. Borrowed from TPS6X
 * \detail
 * \param       unsigned short usCmd
 */
void HID_TcpConnectionCtrl(unsigned short usCmd)
{
    // NOTE: From commands.cpp (user console command), for HID Address and Port change
    //commands, NV param is updated first and then this function will be called. But
    //for HID Connection command, function is called first, then NV param is updated

    if (usCmd & HID_OPEN_TCP_CONN)   /* To open a connection, message from commands.cpp */
    {
        /* Opens a connection Only if previously it was not a TCP connection, then open TCP */
        if (InitialHidConnParam.HIDConnect != HID_CONNECTION_TCP)
        {
            InitialHidConnParam.HIDConnect = HID_CONNECTION_TCP;
            hid_sw_cmd_st.connectionType = HID_CONNECTION_TCP; /* Update local param structure */
            Hid_OpenTcpConnection();
        }
    }
    else if (usCmd & (HID_OPEN_UDP_CONN | HID_NO_CONN | HID_ADDR_CHANGED | HID_PORT_CHANGED | HID_RESET_CONN))
    {
        // Only if the current connection is TCP then kill tcp task.
        if (InitialHidConnParam.HIDConnect == HID_CONNECTION_TCP)
        {
            // There is one chance that the current connection is TCP but the connection is actually not
            //opened, when we had a IP address or Port 0, and then we open TCP connection. In this case, the
            //task wouldn't have created, so killing a non existing task  in FreeRTOS ?
            Hid_CloseTcpConnection(NU_NOT_CONNECTED);

            OsDeleteAdvTask(HIDConnectionTaskPtr);

            /* If Current connection is TCP and Destination address or Port changed, then open new connection */
            if (usCmd & (HID_ADDR_CHANGED | HID_PORT_CHANGED))
            {
                hid_sw_cmd_st.destIPAddress = InitialHidConnParam.HIDAddress;
                hid_sw_cmd_st.destPort = InitialHidConnParam.HIDPort;
                // Open up a new connection
                Hid_OpenTcpConnection();
            }
        }
        // This is from the control system
        if (usCmd & HID_RESET_CONN)
        {
            /* First, copy the values to EEPROM params. Local variables are already up to date */
            InitialHidConnParam.HIDConnect = hid_sw_cmd_st.connectionType;
            InitialHidConnParam.HIDPort = hid_sw_cmd_st.destPort;
            InitialHidConnParam.HIDAddress = hid_sw_cmd_st.destIPAddress;

            if ((hid_sw_cmd_st.connectionType == HID_CONNECTION_TCP) && hid_sw_cmd_st.destIPAddress &&
                hid_sw_cmd_st.destPort)
            {
                Hid_OpenTcpConnection();
            }
        }
        // If the message is for opening up a UDP connection.
        else if (usCmd & HID_OPEN_UDP_CONN)
        {
            InitialHidConnParam.HIDConnect = HID_CONNECTION_UDP;
            hid_sw_cmd_st.connectionType = HID_CONNECTION_UDP;
        }
        // If message is for setting the connection to none
        else if (usCmd & HID_NO_CONN)
        {
            InitialHidConnParam.HIDConnect = HID_CONNECTION_NONE;
            hid_sw_cmd_st.connectionType = HID_CONNECTION_NONE;
        }
        else if (usCmd & (HID_ADDR_CHANGED | HID_PORT_CHANGED))
        {
            hid_sw_cmd_st.destIPAddress = InitialHidConnParam.HIDAddress= pHidConnParam->HIDAddress;
            hid_sw_cmd_st.destPort = InitialHidConnParam.HIDPort= pHidConnParam->HIDPort;
        }
    }

    // feedback
    GetState(STREAM_1)->CresnetSerialMemorize(SYSTEM_SERIAL_USB_HID_CONTROL, GetNetDestination());

    // save the setting in NVL
    TriggerSystemSave();
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Close an existing TCP connection
 * \detail
 * \param       int code
 */
void Hid_CloseTcpConnection(int code)
{
    BOOL abort= false;

    if (code == NU_NOT_CONNECTED)
        abort= true;

    NetClose(g_pstHIDServer->gwSocket, abort);
    //NU_Close_Socket(g_pstHIDServer->gwSocket); /* To be  used when there is a connection */

    g_pstHIDServer->gwSocket = -1;
    g_pstHIDServer->gwStatus = GW_TCP_NOTCONNECTED;

    // If this is already set by send routine, then we clear it after closing connection.
    g_pstHIDServer->gwErrState = 0; //After status is set to NOTCONNECTED, clear the Err State.
    return;
}
/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Opne a new TCP connection
 * \detail
 * \param       void
 */
void Hid_OpenTcpConnection(void)
{
    memset(&g_stHIDServer, 0x00, sizeof (HID_TCP_PARAM_St));
    //Do not open a connection if either the port or address are zero
    if (InitialHidConnParam.HIDPort == 0 || InitialHidConnParam.HIDAddress == 0)
    {
        return;
    }

    /* Read Destination IP address first to a local variable and then copy to structure */
    unsigned long ulHIDAddress =InitialHidConnParam.HIDAddress;
    memcpy((unsigned char*)&g_stHIDServer.gwIPaddress[0],(unsigned char*)&ulHIDAddress,4);

    g_stHIDServer.gwPort     = InitialHidConnParam.HIDPort;
    g_stHIDServer.gwStatus   = GW_TCP_NOTCONNECTED;
    g_stHIDServer.gwSocket   = -1;
    if (HidTcpTxSemaphore == NULL)
    {
        OsCreateLock(&HidTcpTxSemaphore);
    }
    //Create & start reconnect task for this connection
    HIDConnectionTaskPtr=  OsCreateNamedAdvTask(HID_ConnectionTask, HID_TASK_STACK_SIZE ,(UINT32 *)&g_stHIDServer,
                                                HID_TASK_PRIORITY, "HIDCON");
    if (HIDConnectionTaskPtr == NULL)
    {
        DmConsolePrintf("Hid_OpenTcpConnection: FATAL ERROR: Could not create task\r");
    }


    return;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       TCP connection task
 * \detail
 * \param       UINT32 param
 */
void HID_ConnectionTask(UINT32 param)
{
    HID_TCP_PARAM_St *pstHIDServer;
    int result;
    UINT32 IP_Address;

    //This comes as a pointer from the task creation function.
    pstHIDServer = (HID_TCP_PARAM_St*)param;

    while (true)
    {
        if ((pstHIDServer->gwStatus != GW_TCP_CONNECTED))
        {
            if (pstHIDServer->closeReason )
            {
                Hid_CloseTcpConnection(NU_NOT_CONNECTED);
                pstHIDServer->closeReason = ERROR_NOT_CLOSING;
            }

            IP_Address = *((UINT32*)pstHIDServer->gwIPaddress);
            if (IP_Address == 0)
            {
                //pstHIDServer->gwServIpErrCnt++;
            }
            else
            {
                result = NetTcpConnect(IP_Address, pstHIDServer->gwPort, HIDNewConnection,
                                       HIDConn_Receive_Data, (UINT32)pstHIDServer);
                if (result < 0)
                {
                    pstHIDServer->connErrCnt++;
                    pstHIDServer->gwStatus = GW_TCP_NOTCONNECTED;
                    //CloseGatewayConnection(pstHIDServer,ERROR_CONNECT_FAIL,result);
                    DmConsolePrintf("ERROR: HID_ConnectionTask: Failed with code %d\n",result);
                    //socket closed by caller (NetTcpConnect)
                    //NetClose(pstHIDServer->gwSocket,result);
                    pstHIDServer->gwSocket = -1;
                    pstHIDServer->gwStatus = GW_TCP_NOTCONNECTED;
                }
                else
                {
                    pstHIDServer->gwStatus = GW_TCP_CONNECTED;
                    pstHIDServer->connCnt++;
                    //Gateway_Got_Connected(pstHIDServer);
                    //SOme timer need to be started ?? AKN
                }
            }
        }
        else // already connected
        {
            DmConsolePrintf("WARNING: HID_ConnectionTask:Already connected to gateway, nothing to do...\r");
        }
        HwDelayMsec(5000);
    }
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Callback task after connection is succesfull.
 * \detail
 * \param       UINT32 param
 *              UINT32 remoteIp,
 *              UINT16 remotePort
 *              UINT32 socket
 */
INT32 HIDNewConnection(UINT32 param, UINT32 remoteIp, UINT16 remotePort, UINT32 socket)
{
    HID_TCP_PARAM_St *gwThisServer = (HID_TCP_PARAM_St *)param;

    gwThisServer->gwSocket = socket;

    return 0;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Callback task for receiving TCP data.
 * \detail
 * \param       UINT32 param
 *              UINT8  *pData,
 *              UINT32  dataBytes
 */
INT32 HIDConn_Receive_Data(UINT32 param, UINT8 * pData, UINT32 dataBytes)
{
    HID_TCP_PARAM_St *gwThisServer = (HID_TCP_PARAM_St *)param;

    //if(CresnetIsDebug(CRESNET_DEBUG_CIP_PKTS))
    {
        DmConsolePrintf("Received %lu bytes from control system\r", dataBytes);
        DmConsoleHexDumpPacket((UINT8*)pData, (UINT16)dataBytes, "From TCP:", 1);
    }
    if ( dataBytes==0 )
    {
        gwThisServer->closeReason |= ERROR_RECV_FAIL;
        gwThisServer->gwStatus = GW_TCP_NOTCONNECTED;
    }

    //ProcessGatewayPacket(gwThisServer, (char*)pData, dataBytes);
    //return closing flag so the read task can suspend itself
    if ( gwThisServer->closeReason )
        return -1;

    return 0;
}

/**
 * \author      AKN
 * \date        10/26/2008
 * \return      void
 * \retval      none
 * \brief       Initialization. To be called from the Application initialize or some such
 * \detail
 * \param       void
 */
void HID_ConnectionInit(void)
{
    // Global pointer
    g_pstHIDServer = &g_stHIDServer;
    g_pstHIDServer->gwSocket = -1;
    /* Initialize the command structure at startup */
    ////hid_sw_cmd_st.destIPAddress = InitialHidConnParam.HIDAddress;
    ////hid_sw_cmd_st.destPort = InitialHidConnParam.HIDPort;
    ////hid_sw_cmd_st.connectionType = InitialHidConnParam.HIDConnect;
    ////hid_sw_cmd_st.dataTypesDesired = (HID_MOUSE_DATA | HID_KEYBOARD_DATA);  /* By default KB and Mouse enabled */

    ////if (InitialHidConnParam.HIDConnect == HID_CONNECTION_TCP)
    ////{
    ////Hid_OpenTcpConnection();
    ////}
    //Create HID Mouse Repeat Task TBD
    // Create HID Keyboard Repeat task TBD

    //Create Repeat events TBD
    return;
}

BOOL ProcessHIDSerialJoin(const char * msg)
{
    BOOL retval= true;
    //The message is a packet sent as a serial string
    //HID_SWITCH_COMMAND *p_Hid_Sw= (HID_SWITCH_COMMAND *)msg;
    unsigned long ipa;
    unsigned short port;
    unsigned char type;
    unsigned char kbdmse;

    ipa= (msg[3] << 24) | (msg[2] << 16) | (msg[1] << 8) | msg[0];
    port = (msg[5] << 8) | (msg[4]);
    type = (msg[6]);
    kbdmse= (msg[7]);

    if ( (IsDmFieldDebugIndexActive(DM_FIELD_USB_DEBUG_IDX)) )
    {
        DmDebugPrintf("HID:ProcessHIDSerialJoin:%x, %x, %x,%x \r",ipa,port,type, kbdmse);
    }

    //Sanity check
    if ((ipa != 0xFFFFFFFF)  && (port > 4096) &&
        ((type == SEND_HID_NONE) || (type &(SEND_HID_UDP | SEND_HID_TCP))))//Bugzilla 25164
    {
        pHidConnParam->HIDAddress = ipa;
        pHidConnParam->HIDPort    = port;
        //Logic: If ignoreTCP is set, then type being TCP | UDP caused UDP to be set
        // else TCP
        if (type == SEND_HID_NONE)
            pHidConnParam->HIDConnect = HID_CONNECTION_NONE;
        if (type & (SEND_HID_TCP | SEND_HID_UDP))
        {
            if ( (IsDmFieldDebugIndexActive(DM_FIELD_USB_DEBUG_IDX)) )
            {
                DmDebugPrintf("HID:ProcessHIDSerialJoinHere 1\r");
            }

            //Either or both set
            if (((type &SEND_HID_TCP) == SEND_HID_TCP) && ((type &SEND_HID_UDP) == SEND_HID_UDP))
            {
                if ( (IsDmFieldDebugIndexActive(DM_FIELD_USB_DEBUG_IDX)) )
                {
                    DmDebugPrintf("HID:ProcessHIDSerialJoinHere 2\r");
                }

                if (g_bIgnoreTCP)
                    pHidConnParam->HIDConnect = HID_CONNECTION_UDP;
                else
                    pHidConnParam->HIDConnect = HID_CONNECTION_TCP;
            }
            else if ((type &SEND_HID_UDP) == SEND_HID_UDP)
            {
                if ( (IsDmFieldDebugIndexActive(DM_FIELD_USB_DEBUG_IDX)) )
                {
                    DmDebugPrintf("HID:ProcessHIDSerialJoin:HID_CONNECTION_UDP %x, \r",type);
                }

                pHidConnParam->HIDConnect = HID_CONNECTION_UDP;
            }
            else if ((type &SEND_HID_TCP) == SEND_HID_TCP)
            {
                if (g_bIgnoreTCP)
                    return false;
                else
                    pHidConnParam->HIDConnect = HID_CONNECTION_TCP;
            }
            else
            {
                return false;
            }
        }

        if(g_iHID_Sema)
        {
            OsLock(g_iHID_Sema);
        }

        //DisconnectFromHIDTCPServer();
        if ((g_pstHIDServer) && (g_pstHIDServer->gwSocket != -1))
        {
            NetClose(g_pstHIDServer->gwSocket, true);
            g_pstHIDServer->gwSocket= -1;
        }

        if(g_iHID_Sema)
        {
            OsUnlock(g_iHID_Sema);
        }

        TriggerSystemSave();
    }
    else
    {
        retval = false;
        #ifdef DM_V24
        V24ErroLog("ERROR HID Control Serial Join %s\r", msg);
        #else
        DmConsolePrintf("ERROR HID Control Serial Join%s\r", msg);
        #endif
    }

    return retval;
}

/**
 * \author      DRymsza
 * \date        12/16/2010
 * \return      UINT8
 * \retval      number of bytes in HID join
 * \brief       construct SYSTEM_SERIAL_USB_HID_CONTROL join (for feedback reporting)
 * \detail
 * \param       char** pointer to HID join contents
 */
UINT8 GetUSBHIDControlSerialJoin(UINT8 **pSerialString)
{
    // WARNING: we are relaying on the pHidConnParam structure and order

    *pSerialString = (UINT8*)pHidConnParam;

    // 4 - ip address, 2 - port, 1 - connection type, 1 - data type (currently not used)
    return(4+2+1+1);
}

/**
 * \author      AKN
 * \date        10/28/2008
 * \return      int
 * \retval      number of bytes trasnmitted if success, 0,-1 if error
 * \brief       Send to TCP
 * \detail
 * \param       char* buffer, int buffLen
 */
int SendToHIDTCPServer( unsigned char* buffer, int buffLen)
{
    int result= -1;

    if (g_iHID_Sema)
    {
        OsLock(g_iHID_Sema);
    }

    if (InitialHidConnParam.HIDConnect == HID_CONNECTION_TCP)
    {
        if ((g_pstHIDServer->gwStatus == GW_TCP_CONNECTED) && (g_pstHIDServer->gwSocket != -1))
        {
#warning "DO YOU NEED TO USE THE SEMAPHORE??
            result= NetSendTcpData(g_pstHIDServer->gwSocket, buffer, buffLen);
            if (result < 0)
            {
                DmConsolePrintf("SendToHIDTCPServer: Error while sending: code %d\r", result);
            }
        }
    }

    if (g_iHID_Sema)
    {
        OsUnlock(g_iHID_Sema);
    }

    return result;
}

/**
 * \author      AKN
 * \date        10/28/2008
 * \return      int
 * \retval      number of bytes trasnmitted if success, 0,-1 if error
 * \brief       Send to UDP
 * \detail
 * \param       char* buffer, int buffLen
 */
int SendToHIDUDPServer( unsigned char* buffer, int buffLen)
{
    UINT32 ipAddr;
    UINT16 port;
    int result= -1;

    if (g_iHID_Sema)
    {
        OsLock(g_iHID_Sema);
    }



#ifdef DM_V24
    //ProjectEnterMKEvent();//send mouse/keyboard active event to the project
    if (ProjectEnterMKEvent(buffer, buffLen) && pHidConnParam->HIDConnect == HID_CONNECTION_UDP)
#else
    if (pHidConnParam->HIDConnect == HID_CONNECTION_UDP)
#endif
    {
        if (g_pstHIDServer->gwSocket == -1)
        {
            //Create a UDP socket
            if((result = NetCreateUdpSendSocket()) > 0)
            {
                g_pstHIDServer->gwSocket = (int)result;
            }
            else
            {
                DmConsolePrintf("SendToHIDUDPServer: Error while creating socket\r");
                if (g_iHID_Sema)
                {
                    OsUnlock(g_iHID_Sema);
                }
                return result;
            }
        }
        ipAddr= pHidConnParam->HIDAddress;
        port= pHidConnParam->HIDPort;
        if ((ipAddr != 0) && (port != 0))
        {
            result= NetSendUdpData(g_pstHIDServer->gwSocket, buffer, buffLen, ipAddr, port);

            if (result < 0)
            {
                DmConsolePrintf("SendToHIDUDPServer: Error while sending: code %d\r", result);
            }
        }
    }

    if (g_iHID_Sema)
    {
        OsUnlock(g_iHID_Sema);
    }

    return result;
}

/**
 * \author      Chris Merck
 * \date        10 FEB 2011
 * \return      int
 * \retval      number of bytes trasnmitted if success, 0,-1 if error
 * \brief       Send HID Message (inputs)
 * \detail      Based on HIDSendMousePacket, but is now generic
 * \param       char* pBuf, int size // size MUST be 8
 */
INT32 HIDSendMessage(char* pBuf, int size)
{
    HID_PACKET hidMsg;
    int result = -1;
	UINT8 i;

    if (size != 8)
        return result;

    hidMsg.Header.PacketType    = HID_UDPP_MSG;
    hidMsg.Header.Flags         = Swap16(0x0000);
    hidMsg.Header.TransNumber   = Swap16(0x0000);
    hidMsg.Header.DataLength    = Swap16(sizeof(hidMsg) - sizeof(hidMsg.Header));

    memcpy(&hidMsg.Data[0], pBuf, size);

    if ( (IsDmFieldDebugIndexActive(DM_FIELD_USB_DEBUG_IDX)) )
    {
        DmDebugPrintf("HID: Sending Message: ");
		for (i=0;i<size;i++) {
			DmDebugPrintf(" %.2X",pBuf[i]);
		}
		DmDebugPrintf("\r");
    }

    if (InitialHidConnParam.HIDConnect == HID_CONNECTION_TCP)
    {
        result = SendToHIDTCPServer((unsigned char *)&hidMsg, sizeof(hidMsg));
    }
    else
    {
        result = SendToHIDUDPServer((unsigned char *)&hidMsg, sizeof(hidMsg));
    }

    return result;
}

/**
 * \author      Dariusz Rymsza
 * \date        6 Jun 2011
 * \return      void
 * \retval      nothing
 * \brief       Restore default HID parameters (address 0.0.0.0)
 * \param       void
 */
void RestoreDefaultHIDParameters(void)
{
    HIDCONN_PARAM *pTempHidConnParam;   // pointer to working copy for changes

    pTempHidConnParam = GetHidConnParams();
    pTempHidConnParam->HIDAddress = 0;
    pTempHidConnParam->HIDConnect = HID_CONNECTION_UDP;
    pTempHidConnParam->HIDPort = USB_HID_PORT;
}
